<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostLikeTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;

final class PostContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function loadSingle(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postIdFromReq = self::$request -> findKey(PostTable::ID, RequestTable::PAYLOAD, PostTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure target post exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure post owner exists
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks against post owner
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            |  add models to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(PostTable::getTableName(), $targetPostModel -> getDataMap());

            self::addToResponse(UserTable::getTableName(), $postOwnerUserModel -> getDataMap());

            /*
            |--------------------------------------------------------------------------
            |  add additional content
            |--------------------------------------------------------------------------
            */

            /*
            |--------------------------------------------------------------------------
            | additional data | post like map
            | -------------------------------------------------------------------------
            | help build like button
            |--------------------------------------------------------------------------
            */

            self::fetchModelsAndAddAsAdditional(
                PostLikeTable::getTableName(),
                array(
                    PostLikeTable::LIKED_POST_ID    => $targetPostModel -> getId(),
                    PostLikeTable::LIKED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | additional data | post save map
            | -------------------------------------------------------------------------
            | help build save button
            |--------------------------------------------------------------------------
            */

            self::fetchModelsAndAddAsAdditional(
                PostSaveTable::getTableName(),
                array(
                    PostSaveTable::SAVED_POST_ID    => $targetPostModel -> getId(),
                    PostSaveTable::SAVED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );
        });
    }

    /*
    |--------------------------------------------------------------------------
    | feeds processor
    |--------------------------------------------------------------------------
    */

    public static function load(string $feedType, string $loadType): void
    {
        self::feedHelperInit($feedType, $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $offset = self::$request -> findKeyOffset(PostTable::ID, PostTable::TABLE_NAME);

            if (self::isLoadingForProfilePage())
            {
                $userIdFromReq = self::$request -> findKey(
                    PostTable::OWNER_USER_ID,
                    RequestTable::PAYLOAD,
                    PostTable::TABLE_NAME
                );

                self::ensureValue($userIdFromReq);
            }

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | if profile feeds, make sure posts are not privacy protected
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingForProfilePage())
            {
                /*
                |--------------------------------------------------------------------------
                |  get target user
                |--------------------------------------------------------------------------
                */

                $targetUserModel = UserModel::findFromId_throwException($userIdFromReq);

                /*
                |--------------------------------------------------------------------------
                | posts privacy checks
                |--------------------------------------------------------------------------
                */

                if ( ! UserModelHelper::isUserContentAvailable($targetUserModel, self::$authedUserModel))
                {
                    throw new RequestException(ResponseConstants::PRIVATE_RESULTS_MSG);
                }
            }

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $postTableQuery = (new Query()) -> from(PostTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $postTableQuery -> greaterThan(PostTable::ID, $offset);
            }
            else
            {
                $postTableQuery -> lessThan(PostTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | if news feeds
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingForNewsFeedsPage())
            {
                /*
                |--------------------------------------------------------------------------
                | fetch user's following list if 'news feeds'
                |--------------------------------------------------------------------------
                */

                // ? I think we can optimize this call
                //
                $followingUserIds = CommonQueries::userFollowingIds(self::$authedUserModel -> getId());

                /*
                |--------------------------------------------------------------------------
                | show posts from yourself as well
                |--------------------------------------------------------------------------
                */

                $followingUserIds[] = self::$authedUserModel -> getId();

                /*
                |--------------------------------------------------------------------------
                | create In clause
                |--------------------------------------------------------------------------
                */

                $postTableQuery -> in(PostTable::OWNER_USER_ID, $followingUserIds);
            }
            /*
            |--------------------------------------------------------------------------
            | else if profile feeds, just fetch post from target user
            |--------------------------------------------------------------------------
            */
            else
            {
                $postTableQuery -> where(PostTable::OWNER_USER_ID, $userIdFromReq);
            }

            /*
            |--------------------------------------------------------------------------
            | order by post id & limit
            |--------------------------------------------------------------------------
            */

            $postTableQuery
                -> orderByDesc(PostTable::ID)
                -> limit(
                    Settings::getString(
                        self::isLoadingForNewsFeedsPage()
                        ? ServerConstants::SS_INT_LIMIT_LOAD_POST_NEWS_FEEDS
                        : ServerConstants::SS_INT_LIMIT_LOAD_POST_PROFILE
                    )
                );

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $postBeans = $postTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($postBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | prepare maps
            |--------------------------------------------------------------------------
            */

            self::processBeans(PostTable::getTableName(), $postBeans, function (PostModel $model)
            {
                /*
                |--------------------------------------------------------------------------
                | list of users(who's posts are selected)
                |--------------------------------------------------------------------------
                */

                self::addDependency(UserTable::getTableName(), $model -> getOwnerUserId());
            });

            /*
            |--------------------------------------------------------------------------
            | process dependencise
            |--------------------------------------------------------------------------
            */

            self::processDependencies();

            /*
            |--------------------------------------------------------------------------
            | additional data | user follow maps
            | -------------------------------------------------------------------------
            | help build follow button
            |--------------------------------------------------------------------------
            */

            $usersContainer = self::$dataDock -> getContainer(UserTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                UserFollowTable::getTableName(),
                array(
                    UserFollowTable::FOLLOWED_USER_ID    => $usersContainer -> getIds(),
                    UserFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | additional data | post likes maps
            | -------------------------------------------------------------------------
            | help build like button
            |--------------------------------------------------------------------------
            */

            $postsContainer = self::$dataDock -> getContainer(PostTable::getTableName());

            self::fetchModelsAndAddAsAdditional(
                PostLikeTable::getTableName(),
                array(
                    PostLikeTable::LIKED_POST_ID    => $postsContainer -> getIds(),
                    PostLikeTable::LIKED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | additional data | post save maps
            | -------------------------------------------------------------------------
            | help build post save button
            |--------------------------------------------------------------------------
            */

            self::fetchModelsAndAddAsAdditional(
                PostSaveTable::getTableName(),
                array(
                    PostSaveTable::SAVED_POST_ID    => $postsContainer -> getIds(),
                    PostSaveTable::SAVED_BY_USER_ID => self::$authedUserModel -> getId(),
                )
            );
        });
    }

    /*
    |--------------------------------------------------------------------------
    | infinite feeds proprocessor
    |--------------------------------------------------------------------------
    */

    public static function inifinite(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $offset = self::$request -> findKeyOffset(PostTable::ID, PostTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | for some magic, put it in a while
            |--------------------------------------------------------------------------
            */

            while (true)
            {
                /*
                |--------------------------------------------------------------------------
                | query builder
                |--------------------------------------------------------------------------
                */

                $postTableQuery = (new Query()) -> from(PostTable::TABLE_NAME);

                /*
                |--------------------------------------------------------------------------
                | add offset
                |--------------------------------------------------------------------------
                */

                if (self::isAvailable($offset) && '0' != $offset)
                {
                    $postTableQuery -> lessThan(PostTable::ID, $offset);
                }

                /*
                |--------------------------------------------------------------------------
                | ignore current user's posts
                |--------------------------------------------------------------------------
                */

                $postTableQuery -> whereNot(PostTable::OWNER_USER_ID, self::$authedUserModel -> getId());

                /*
                |--------------------------------------------------------------------------
                | order by post id & limit
                |--------------------------------------------------------------------------
                */

                $postTableQuery
                    -> orderByDesc(PostTable::ID)
                    -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_POST_INFINITE));

                $postBeans = $postTableQuery -> select();

                /*
                |--------------------------------------------------------------------------
                | check end of results
                |--------------------------------------------------------------------------
                */

                if (0 == \count($postBeans))
                {
                    return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
                }

                /*
                |--------------------------------------------------------------------------
                | prepare fetched user ids and post ids
                |--------------------------------------------------------------------------
                */

                $userIdToPostIdsMap = array();

                /**
                 * @var PostModel[]
                 */
                $fetchedPostModels = array();

                /**
                 * @var UserModel[]
                 */
                $fetchedUserModels = array();

                foreach ($postBeans as $postBean)
                {
                    $postModel = PostModel::createFromUntouchedBean_noException($postBean);

                    if ($postModel -> isModel())
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | add post model to list
                        |--------------------------------------------------------------------------
                        */

                        $fetchedPostModels[] = $postModel;

                        /*
                        |--------------------------------------------------------------------------
                        | get ids
                        |--------------------------------------------------------------------------
                        */

                        $postId = $postModel -> getId();

                        $userId = $postModel -> getOwnerUserId();

                        /*
                        |--------------------------------------------------------------------------
                        | update offset, move to lower id if possible
                        |--------------------------------------------------------------------------
                        */

                        if ($offset > $postId)
                        {
                            $offset = $postId;
                        }

                        /*
                        |--------------------------------------------------------------------------
                        | update user id => post ids map
                        |--------------------------------------------------------------------------
                        */

                        if ( ! isset($userIdToPostIdsMap[$userId]))
                        {
                            $userIdToPostIdsMap[$userId] = array($postId);

                            continue;
                        }

                        $userIdToPostIdsMap[$userId][] = $postId;
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | fetch users first, do pirvacy checks here
                |--------------------------------------------------------------------------
                */

                /**
                 * @var UserModel[]
                 */
                $userModelsFromQuery = CommonQueries::modelsWithMatchingPredicates(
                    UserTable::getTableName(),
                    array(
                        UserTable::ID => \array_keys($userIdToPostIdsMap),
                    )
                );

                foreach ($userModelsFromQuery as $userModel)
                {
                    if (UserModelHelper::isUserContentAvailable($userModel, self::$authedUserModel))
                    {
                        $fetchedUserModels[] = $userModel;
                    }
                    else
                    {
                        // remove user content from results

                        unset($userIdToPostIdsMap[$userModel -> getId()]);
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | whether after privacy checks we've something to respond or not?
                |--------------------------------------------------------------------------
                */

                if (0 == \count($fetchedUserModels))
                {
                    // do nothing, let the while loop run and try adding something to content
                }
                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | add users to response
                    |--------------------------------------------------------------------------
                    */

                    foreach ($fetchedUserModels as $userModel)
                    {
                        self::addToResponse(UserTable::getTableName(), $userModel -> getDataMap());
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | add posts to response
                    |--------------------------------------------------------------------------
                    */

                    foreach ($fetchedPostModels as $postModel)
                    {
                        self::addToResponse(PostTable::getTableName(), $postModel -> getDataMap());
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | additional data | user follow maps
                    | -------------------------------------------------------------------------
                    | help build follow button
                    |--------------------------------------------------------------------------
                    */

                    $usersContainer = self::$dataDock -> getContainer(UserTable::getTableName());

                    self::fetchModelsAndAddAsAdditional(
                        UserFollowTable::getTableName(),
                        array(
                            UserFollowTable::FOLLOWED_USER_ID    => $usersContainer -> getIds(),
                            UserFollowTable::FOLLOWED_BY_USER_ID => self::$authedUserModel -> getId(),
                        )
                    );

                    /*
                    |--------------------------------------------------------------------------
                    | additional data | post likes maps
                    | -------------------------------------------------------------------------
                    | help build like button
                    |--------------------------------------------------------------------------
                    */

                    $postsContainer = self::$dataDock -> getContainer(PostTable::getTableName());

                    self::fetchModelsAndAddAsAdditional(
                        PostLikeTable::getTableName(),
                        array(
                            PostLikeTable::LIKED_POST_ID    => $postsContainer -> getIds(),
                            PostLikeTable::LIKED_BY_USER_ID => self::$authedUserModel -> getId(),
                        )
                    );

                    /*
                    |--------------------------------------------------------------------------
                    | additional data | post save maps
                    | -------------------------------------------------------------------------
                    | help build post save button
                    |--------------------------------------------------------------------------
                    */

                    self::fetchModelsAndAddAsAdditional(
                        PostSaveTable::getTableName(),
                        array(
                            PostSaveTable::SAVED_POST_ID    => $postsContainer -> getIds(),
                            PostSaveTable::SAVED_BY_USER_ID => self::$authedUserModel -> getId(),
                        )
                    );

                    /*
                    |--------------------------------------------------------------------------
                    | break the loop
                    |--------------------------------------------------------------------------
                    */

                    break;
                }
            }
        });
    }
}
